/**
* \file: UspiInputSourceImpl.cpp
*
* \version: 2.0
*
* \release: $Name:$
*
* Actual implementation of UspiInputSource class (see pImpl idiom)
*
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*          P. Acar / ADIT/ESM / pacar@de.adit-jv.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <string>
#include <adit_logging.h>
#include <uspi/WaylandFacade.h>
#include <uspi/ITouchFacade.h>
#include <bdcl/CCarLifeLib.h>
#include "UspiInputSourceImpl.h"

LOG_IMPORT_CONTEXT(bdcl_input)

namespace adit { namespace bdcl {

using namespace uspi;
using std::string;
using std::unique_ptr;

UspiInputSourceImpl::UspiInputSourceImpl(IAditInputSourceCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
: mSessionContext(inCallbackDealer), mCallbacks(inCallbacks), mTouchFacade(nullptr), mRunning(false), mTrackedTouchX(0),
  mTrackedTouchY(0) { }

UspiInputSourceImpl::~UspiInputSourceImpl()
{
    if (true == mRunning)
    {
        teardown();
    }
}

void UspiInputSourceImpl::setConfigItem(std::string inKey, std::string inValue)
{
    inKey = inKey;
    inValue = inValue;
}

bool UspiInputSourceImpl::initialize()
{
    mTouchFacade = move(unique_ptr<WaylandFacade>((new WaylandFacade(mSessionContext))->
            setDisplaySize(1, 1)-> // we do not need any scaling by USPI touch for the touch points
                                   // therefore we set 1 for the display size. This means that the
                                   // touch points are divided by "1" and we get the touch points in
                                   // display coordinates.
            setFingerNum(1)));

    if(!mTouchFacade->initialize(this))
    {
        LOG_ERROR((bdcl_input, "Failed to initialize touch facade"));
        return false;
    }

    mRunning = true;

    return true;
}

void UspiInputSourceImpl::teardown()
{
    if (mRunning) {
        if (!mTouchFacade->deinitialize())
        {
            LOG_ERROR((bdcl_input, "Failed to deinitialize touch facade"));
        } else {
            mRunning = false;
        }
    } else {
        LOGD_DEBUG((bdcl_input, "UspiInputSourceImpl::shutdown()  Not running. Was already stopped"));
    }

    mTouchFacade = nullptr;

    LOGD_DEBUG((bdcl_input, "UspiInputSourceImpl shutdown done"));
}

void UspiInputSourceImpl::onTouch(TouchEvent inEvent)
{
    LOGD_VERBOSE((bdcl_input, "entry: type: %d, finger: %d, x: %f, y: %f",
              inEvent.eventType, inEvent.fingerId, inEvent.xRelative, inEvent.yRelative));

    S_TOUCH_ACTION outAction;

    int x_pixel = inEvent.xRelative;
    int y_pixel = inEvent.yRelative;

    switch (inEvent.eventType)
    {
    case TouchEvent::down:
        LOGD_VERBOSE((bdcl_input, "Touch down event"));

        outAction.action = 0;
        outAction.x = x_pixel;
        outAction.y = y_pixel;
        CCarLifeLib::getInstance()->ctrlTouchAction(&outAction);

        mTrackedTouchX = x_pixel;
        mTrackedTouchY = y_pixel;
        break;

    case TouchEvent::move:
        LOGD_VERBOSE((bdcl_input, "Touch move event"));

        outAction.action = 2;
        outAction.x = x_pixel;
        outAction.y = y_pixel;
        CCarLifeLib::getInstance()->ctrlTouchAction(&outAction);

        mTrackedTouchX = x_pixel;
        mTrackedTouchY = y_pixel;
        break;

    case TouchEvent::up:
        LOGD_VERBOSE((bdcl_input, "Touch up event"));

        outAction.action = 1;
        outAction.x = mTrackedTouchX;
        outAction.y = mTrackedTouchY;
        CCarLifeLib::getInstance()->ctrlTouchAction(&outAction);
        break;
    default:
        break;
    }
}

void UspiInputSourceImpl::onTouchError(int inError)
{
    // todo get rid of this
    LOG_ERROR((bdcl_input, "Check Error: %d", inError));
    mCallbacks->onError((bdclErrorCodes)inError);
}

void UspiInputSourceImpl::onLogging(UspiLogLevel inLogLevel, const string& inLogString)
{
    switch (inLogLevel) {
        case USPI_LOG_FATAL:
            LOG_FATAL((bdcl_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_ERROR:
            LOG_ERROR((bdcl_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_WARN:
            LOG_WARN((bdcl_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_INFO:
            LOG_INFO((bdcl_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_DEBUG:
            LOGD_DEBUG((bdcl_input, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_VERBOSE:
            LOGD_VERBOSE((bdcl_input, "%s", inLogString.c_str()));
            break;
        default:
            break;
    }
}

} } /* namespace adit { namespace bdcl { */

